<?php
session_start();
if (!isset($_SESSION['student_id']) || !isset($_SESSION['exam_id'])) {
    header('Location: student_login.php');
    exit;
}
$student_id = $_SESSION['student_id'];
$student_name = isset($_SESSION['student_name']) ? $_SESSION['student_name'] : 'Student';
$exam_id = $_SESSION['exam_id'];

require_once 'connectiondb.php';
try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $user, $pass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die("Database connection failed: " . $e->getMessage());
}

// Check if student is assigned the exam and not completed
try {
    $stmt = $pdo->prepare("SELECT status FROM student_exams WHERE student_id = ? AND exam_id = ?");
    $stmt->execute([$student_id, $exam_id]);
    $exam_status = $stmt->fetchColumn();
} catch (PDOException $e) {
    die("Failed to check exam assignment: " . $e->getMessage());
}

if ($exam_status === false) {
    header('Location: student_dashboard.php?error=You are not assigned this exam.');
    exit;
} elseif ($exam_status === 'completed') {
    header('Location: exam_completed.php');
    exit;
}

// Fetch exam duration
try {
    $stmt = $pdo->prepare("SELECT duration FROM exams WHERE id = ?");
    $stmt->execute([$exam_id]);
    $exam_duration = $stmt->fetchColumn();
    if (!$exam_duration) {
        $exam_duration = 30; // default duration in minutes if not set
    }
} catch (PDOException $e) {
    die("Failed to fetch exam duration: " . $e->getMessage());
}

// Fetch questions and options for the exam
try {
    $stmt = $pdo->prepare("
        SELECT q.id AS question_id, q.question_text, o.id AS option_id, o.option_text
        FROM questions q
        LEFT JOIN options o ON q.id = o.question_id
        WHERE q.exam_id = ?
        ORDER BY q.id, o.id
    ");
    $stmt->execute([$exam_id]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die("Failed to fetch questions: " . $e->getMessage());
}

if (empty($rows)) {
    die("No questions found for this exam.");
}

$questions = [];
foreach ($rows as $row) {
    $qid = $row['question_id'];
    if (!isset($questions[$qid])) {
        $questions[$qid] = [
            'question_text' => $row['question_text'],
            'options' => []
        ];
    }
    if ($row['option_id'] !== null) {
        $questions[$qid]['options'][] = [
            'option_id' => $row['option_id'],
            'option_text' => $row['option_text']
        ];
    }
}

// Shuffle questions and options
$question_keys = array_keys($questions);
shuffle($question_keys);
$shuffled_questions = [];
foreach ($question_keys as $key) {
    $options = $questions[$key]['options'];
    shuffle($options);
    $shuffled_questions[$key] = [
        'question_text' => $questions[$key]['question_text'],
        'options' => $options
    ];
}
$total_questions = count($shuffled_questions);
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1, minimum-scale=0.5, maximum-scale=1.0, user-scalable=yes" />
<title>Take Exam - MCQ Exam Portal</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
<style>
.question-nav-btn.attempted {background-color: green !important; color: white !important;}
.question-nav-btn.not-attempted {background-color: red !important; color: white !important;}
.question-nav-btn.skipped {background-color: violet !important; color: white !important;}
.question-nav-btn {margin: 2px; padding: 5px 10px; border: none; border-radius: 4px;}
.question {margin-bottom: 20px;}
.option {
    border: 1px solid #ccc;
    border-radius: 6px;
    padding: 15px 20px;
    margin-bottom: 15px;
    cursor: pointer;
    user-select: none;
    transition: background-color 0.3s ease, border-color 0.3s ease, box-shadow 0.3s ease;
    box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    background-color: #fff;
    color: #333;
    font-size: 16px;
}
.option:hover {
    background-color: #e9ecef;
    box-shadow: 0 4px 10px rgba(0,0,0,0.15);
}
.option.selected {
    background-color: #c3e6cb;
    border-color: #4caf50;
    box-shadow: 0 4px 12px rgba(76, 175, 80, 0.6);
    color: #2e7d32;
    font-weight: 600;
}
#studentCamera {
    position: fixed; 
    bottom : 10px; 
    right: 10px; 
    width: 160px; 
    height: 120px; 
    border: 2px solid #2980b9; 
    border-radius: 8px; 
    z-index: 1000; 
    background-color: black; 
    resize: both; 
    overflow: auto; 
    cursor: move;
}
#selected-question-container {
    width: 100%;
    height: 50vh;
    overflow-y: auto;
}
.question {
    width: 100%;
    height: 50vh;
    overflow-y: auto;
}
.d-flex.gap-2.mb-4 {
    margin-top: 10px;
    z-index: 10;
    position: relative;
}
</style>
</head>
<body>
<nav class="navbar navbar-expand-lg navbar-dark bg-primary mb-4">
<div class="container-fluid">
<a class="navbar-brand" href="#">MCQ Exam Portal</a>
<button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav" 
aria-controls="navbarNav" aria-expanded="false" aria-label="Toggle navigation">
<span class="navbar-toggler-icon"></span>
</button>
<div class="collapse navbar-collapse" id="navbarNav">
<ul class="navbar-nav ms-auto">
  
<li class="nav-item"><span class="nav-link disabled">Student: <?php echo htmlspecialchars($student_name); ?> (ID: <?php echo htmlspecialchars($student_id); ?>)</span></li>
<li class="nav-item"><a class="nav-link" href="student_logout.php">Logout</a></li>
</ul>
</div>
</div>
</nav>

<!-- Camera video element -->
<video id="studentCamera" autoplay muted playsinline></video>

<div class="container">
      <div id="google_translate_element"></div>

<script type="text/javascript">
function googleTranslateElementInit() {
  new google.translate.TranslateElement({pageLanguage: 'en'}, 'google_translate_element');
}
</script>

<script type="text/javascript" src="//translate.google.com/translate_a/element.js?cb=googleTranslateElementInit"></script>
<h2 class="mb-3">Exam</h2>
<div id="timer" class="fw-bold fs-5 mb-3"></div>
<div>Total Questions: <?php echo $total_questions; ?></div>
<form id="examForm" method="POST" action="submit_answers.php">
<div id="selected-question-container" class="mb-4">
<?php 
$questionIndex = 0;
foreach ($shuffled_questions as $qid => $q): ?>
<div class="question" data-question-index="<?php echo $questionIndex; ?>" <?php if($questionIndex === 0) { echo ''; } else { echo 'style="display:none;"'; } ?>>
<p><strong><?php echo htmlspecialchars($q['question_text']); ?></strong></p>
<?php 
$optionLabels = ['a', 'b', 'c', 'd'];
foreach ($q['options'] as $index => $option): ?>
<div class="option" tabindex="0" data-option-id="<?php echo htmlspecialchars($option['option_id']); ?>" data-question-id="<?php echo htmlspecialchars($questionIndex); ?>">
<span style="font-weight:bold; margin-right:5px;"><?php echo htmlspecialchars($optionLabels[$index] ?? ''); ?>.</span>
<?php echo htmlspecialchars($option['option_text']); ?>
</div>
<?php endforeach; ?>
<input type="hidden" name="answers[<?php echo htmlspecialchars($qid); ?>]" id="answer-<?php echo htmlspecialchars($questionIndex); ?>" value="">
</div>
<?php 
$questionIndex++;
endforeach; ?>
</div>
<div id="question-navigation" class="mb-3">
<?php 
$questionIndex = 0;
foreach ($shuffled_questions as $qid => $q): ?>
<button type="button" class="question-nav-btn not-attempted" data-question-index="<?php echo $questionIndex; ?>">
<?php echo $questionIndex + 1; ?>
</button>
<?php 
$questionIndex++;
endforeach; ?>
</div>
<div class="d-flex gap-2 mb-4">
<button type="button" id="prevBtn" class="btn btn-secondary" disabled>Previous</button>
<button type="button" id="nextBtn" class="btn btn-primary" disabled>Next</button>
<button type="button" id="skipBtn" class="btn btn-warning">Skip</button>
<button type="button" id="finalSubmitBtn" class="btn btn-danger ms-auto" disabled>Final Submit</button>
</div>
<div class="modal fade" id="submitModal" tabindex="-1" aria-labelledby="submitModalLabel" aria-hidden="true">
<div class="modal-dialog modal-dialog-centered">
<div class="modal-content">
<div class="modal-header">
<h5 class="modal-title" id="submitModalLabel">Confirm Submission</h5>
<button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
</div>
<div class="modal-body">
<p>Attempted Questions: <span id="attemptedCount">0</span></p>
<p>Skipped Questions: <span id="skippedCount">0</span></p>
<p>Not Attempted Questions: <span id="notAttemptedCount">0</span></p>
</div>
<div class="modal-footer">
<button type="button" id="confirmSubmit" class="btn btn-danger">Submit</button>
<button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
</div>
</div>
</div>
</div>
</form>
</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
let timeLeft = <?php echo intval($exam_duration) * 60; ?>; // duration in seconds
function startTimer() {
    const timerDisplay = document.getElementById('timer');
    const form = document.getElementById('examForm');
    const interval = setInterval(() => {
        let minutes = Math.floor(timeLeft / 60);
        let seconds = timeLeft % 60;
        timerDisplay.textContent = `Time Left: ${minutes}m ${seconds}s`;
        if (timeLeft <= 0) {
            clearInterval(interval);
            alert('Time is up! Your exam will be submitted automatically.');
            form.submit();
        }
        timeLeft--;
    }, 1000);
}

function handleVisibilityChange() {
    const form = document.getElementById('examForm');
    if (document.hidden) {
        alert('You switched tabs or minimized the window. Your exam will be submitted automatically.');
        form.submit();
    }
}

document.addEventListener('keydown', function(e) {
    if (e.key === 'PrintScreen') {
        alert('Screen capture is not allowed during the exam.');
        navigator.clipboard.writeText('');
    }
});

window.addEventListener('blur', function() {
    console.log('Window lost focus - possible screen capture attempt.');
});

document.addEventListener('visibilitychange', handleVisibilityChange);

document.addEventListener('contextmenu', function(e) { e.preventDefault(); });

document.addEventListener('keydown', function(e) {
    if ((e.ctrlKey || e.metaKey) && (e.key === 'c' || e.key === 'x' || e.key === 'v')) {
        e.preventDefault();
        alert('Copy, cut, and paste are disabled during the exam.');
    }
});

document.addEventListener('selectstart', function(e) { e.preventDefault(); });

window.onload = async function() {
    console.log('Window loaded');
    startTimer();
    await initializeExam();
    startCameraMandatory();
};

let cameraAccessGranted = false;
function startCameraMandatory() {
    console.log('startCameraMandatory called');
    const video = document.getElementById('studentCamera');
    const finalSubmitBtn = document.getElementById('finalSubmitBtn');
    const prevBtn = document.getElementById('prevBtn');
    const nextBtn = document.getElementById('nextBtn');

    if (navigator.mediaDevices && navigator.mediaDevices.getUserMedia) {
        navigator.mediaDevices.getUserMedia({ video: { facingMode: { ideal: "user" } } })
            .then(stream => {
                console.log('Camera stream started');
                video.srcObject = stream;
                cameraAccessGranted = true;
                if (finalSubmitBtn) finalSubmitBtn.disabled = false;
                if (prevBtn) prevBtn.disabled = false;
                if (nextBtn) nextBtn.disabled = false;
            })
            .catch(err => {
                alert('Camera access is mandatory to take the exam. Please allow camera access.');
                cameraAccessGranted = false;
                if (finalSubmitBtn) finalSubmitBtn.disabled = true;
                if (prevBtn) prevBtn.disabled = true;
                if (nextBtn) nextBtn.disabled = true;
            });
    } else {
        alert('Camera API is not supported in this browser.');
        cameraAccessGranted = false;
        if (finalSubmitBtn) finalSubmitBtn.disabled = true;
        if (prevBtn) prevBtn.disabled = true;
        if (nextBtn) nextBtn.disabled = true;
    }

    const form = document.getElementById('examForm');
    if (form) {
        form.addEventListener('submit', function(e) {
            if (!cameraAccessGranted) {
                e.preventDefault();
                alert('You must allow camera access to submit your answers.');
            }
        });
    }
}

/* Removed window resize event listener to allow screen resizing */
/*
window.addEventListener('resize', function() {
    const minWidth = 800;
    const minHeight = 600;
    const form = document.getElementById('examForm');
    if (window.innerWidth < minWidth || window.innerHeight < minHeight) {
        alert('Split screen or window resizing is not allowed during the exam. Please resize your window to full screen. Your exam will be submitted automatically if you do not comply.');
        form.submit();
    }
});
*/

// Disable auto-submit on scroll by ignoring scroll events
window.addEventListener('scroll', function(e) {
    // Do nothing on scroll to prevent auto-submit
});

async function initializeExam() {
    console.log('initializeExam called');
    const questions = document.querySelectorAll('#selected-question-container .question');
    const navButtons = document.querySelectorAll('.question-nav-btn');
    const prevBtn = document.getElementById('prevBtn');
    const nextBtn = document.getElementById('nextBtn');
    const finalSubmitBtn = document.getElementById('finalSubmitBtn');
    const skipBtn = document.getElementById('skipBtn');
    const submitModal = new bootstrap.Modal(document.getElementById('submitModal'));
    let currentQuestionIndex = 0;
    let answerStatus = Array(questions.length).fill('not-attempted');

    function updateNavButtonColors() {
        console.log('updateNavButtonColors called');
        navButtons.forEach((btn, i) => {
            btn.classList.remove('attempted', 'not-attempted', 'skipped');
            btn.classList.add(answerStatus[i]);
        });
    }

    async function showQuestionByIndex(index) {
        console.log('showQuestionByIndex called with index:', index);
        questions.forEach((q, i) => {
            q.style.display = 'none';
        });
        const questionToShow = questions[index];
        if (questionToShow) {
            questionToShow.style.display = 'block';
        }
        navButtons.forEach((btn, i) => {
            if (i === index) {
                btn.classList.add('active');
            } else {
                btn.classList.remove('active');
            }
        });
        prevBtn.disabled = index === 0;
        if (index === questions.length - 1) {
            nextBtn.textContent = 'Submit Answers';
        } else {
            nextBtn.textContent = 'Next';
        }
        currentQuestionIndex = index;
        checkAnswerSelected();
        updateNavButtonColors();
    }

    function checkAnswerSelected() {
        console.log('checkAnswerSelected called');
        const currentQuestion = questions[currentQuestionIndex];
        const selectedOptionId = document.getElementById('answer-' + currentQuestionIndex).value;
        if (selectedOptionId) {
            nextBtn.disabled = false;
            answerStatus[currentQuestionIndex] = 'attempted';
        } else if (answerStatus[currentQuestionIndex] !== 'skipped') {
            nextBtn.disabled = true;
            answerStatus[currentQuestionIndex] = 'not-attempted';
        }
        updateNavButtonColors();
        // Enable final submit button regardless of completion status
        finalSubmitBtn.disabled = false;
    }

    const options = document.querySelectorAll('.option');
    options.forEach(option => {
        option.addEventListener('click', function(e) {
            console.log('option clicked:', option);
            const questionId = option.getAttribute('data-question-id');
            const optionId = option.getAttribute('data-option-id');
            const questionDiv = option.closest('.question');
            if (!questionDiv) return;
            const optionsInQuestion = questionDiv.querySelectorAll('.option');
            optionsInQuestion.forEach(opt => opt.classList.remove('selected'));
            option.classList.add('selected');
            const hiddenInput = document.getElementById('answer-' + questionId);
            if (hiddenInput) {
                hiddenInput.value = optionId;
            }
            checkAnswerSelected();
        });
    });

    navButtons.forEach((btn, index) => {
        btn.addEventListener('click', () => {
            console.log('nav button clicked:', index);
            showQuestionByIndex(index);
        });
    });

    prevBtn.addEventListener('click', () => {
        console.log('prevBtn clicked');
        if (currentQuestionIndex > 0) {
            showQuestionByIndex(currentQuestionIndex - 1);
        }
    });

    nextBtn.addEventListener('click', () => {
        console.log('nextBtn clicked');
        if (currentQuestionIndex === questions.length - 1) {
            showSubmitModal();
        } else {
            showQuestionByIndex(currentQuestionIndex + 1);
        }
    });

    skipBtn.addEventListener('click', () => {
        console.log('skipBtn clicked');
        answerStatus[currentQuestionIndex] = 'skipped';
        updateNavButtonColors();
        if (currentQuestionIndex === questions.length - 1) {
            showSubmitModal();
        } else {
            showQuestionByIndex(currentQuestionIndex + 1);
        }
    });

    function showSubmitModal() {
        console.log('showSubmitModal called');
        let attemptedCount = answerStatus.filter(status => status === 'attempted').length;
        let skippedCount = answerStatus.filter(status => status === 'skipped').length;
        let notAttemptedCount = answerStatus.length - attemptedCount - skippedCount;
        document.getElementById('attemptedCount').textContent = attemptedCount;
        document.getElementById('skippedCount').textContent = skippedCount;
        document.getElementById('notAttemptedCount').textContent = notAttemptedCount;
        submitModal.show();
    }

    document.getElementById('confirmSubmit').addEventListener('click', function() {
        console.log('confirmSubmit clicked');
        submitModal.hide();
        document.getElementById('examForm').submit();
    });

    finalSubmitBtn.addEventListener('click', function() {
        console.log('finalSubmitBtn clicked');
        showSubmitModal();
    });

    // Camera access mandatory
    let cameraAccessGranted = false;
    function startCameraMandatory() {
        console.log('startCameraMandatory called');
        const video = document.getElementById('studentCamera');
        const finalSubmitBtn = document.getElementById('finalSubmitBtn');
        const prevBtn = document.getElementById('prevBtn');
        const nextBtn = document.getElementById('nextBtn');

        if (navigator.mediaDevices && navigator.mediaDevices.getUserMedia) {
            navigator.mediaDevices.getUserMedia({ video: { facingMode: { ideal: "user" } } })
                .then(stream => {
                    console.log('Camera stream started');
                    video.srcObject = stream;
                    cameraAccessGranted = true;
                    if (finalSubmitBtn) {
                        finalSubmitBtn.disabled = false;
                    }
                    if (prevBtn) {
                        prevBtn.disabled = false;
                    }
                       if (nextBtn) {
                        nextBtn.disabled = false;
                    }
                })
                .catch(err => {
                    alert('Camera access is mandatory to take the exam. Please allow camera access.');
                    cameraAccessGranted = false;
                    if (finalSubmitBtn) {
                        finalSubmitBtn.disabled = true;
                    }
                    if (prevBtn) {
                        prevBtn.disabled = true;
                    }
                    if (nextBtn) {
                        nextBtn.disabled = true;
                    }
                });
        } else {
            alert('Camera API is not supported in this browser.');
            cameraAccessGranted = false;
            if (finalSubmitBtn) {
                finalSubmitBtn.disabled = true;
            }
            if (prevBtn) {
                prevBtn.disabled = true;
            }
            if (nextBtn) {
                nextBtn.disabled = true;
            }
        }

        const form = document.getElementById('examForm');
        if (form) {
            form.addEventListener('submit', function(e) {
                if (!cameraAccessGranted) {
                    e.preventDefault();
                    alert('You must allow camera access to submit your answers.');
                }
            });
        }
    }

    startCameraMandatory();
}
</script>
</body>
</html>
